<?php

use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\ServerException;

require_once __DIR__ . '/alias_prev.php';

require_once __DIR__ . '/alias_comunas.php';

require_once __DIR__ . '/alias_paises.php';

require_once __DIR__ . '/../inc/vendor/autoload.php';

require_once __DIR__ . '/../lib/variables_entorno/variables_entorno.php';

date_default_timezone_set('America/Santiago');

$url_api = 'https://tomademuestras.api.openagora.cl';

function esDerivada($hoja) {
	$fila = 2;
	
	$celda = $hoja->getCell('L' . $fila);
	$valor = $celda->getValue();
	
	if ($valor === null || trimPersonalizado($valor) === '') {
		return false;
	}
	
	return true;
}

function obtenerCantidadPacientes($hoja) {
	$fila = 2;
	while (true) {
		$celda = $hoja->getCell('A' . $fila);
		$valor = $celda->getValue();
		if ($valor === null || trimPersonalizado($valor) === '') break;
		$fila++;
	}
	
	return $fila - 2;
}

function trimPersonalizado($texto) {
	$texto = trim($texto);
	$texto = trim($texto, chr(0xC2) . chr(0xA0));
	
	return $texto;
}

function limpiarRut($rut) {	
	$rut = preg_replace('/[^k\-0-9]/i', '', $rut);
	$rut = mb_strtoupper($rut, 'UTF-8');
	return $rut;
}

function validarSintaxisRut($rut) {
	$resultado = preg_match('/^([0-9]+)-([0-9]|K)$/', $rut);
	if ($resultado === 1) {
		return true;		
	}
	return false;
}

function validarRut($rut) {
  $dv  = substr($rut, -1);
  $numero = substr($rut, 0, strlen($rut) - 2);
  $i = 2;
  $suma = 0;

  foreach (array_reverse(str_split($numero)) as $v) {
    if ($i == 8) $i = 2;
    $suma += $v * $i;
    $i++;
  }

  $dvr = 11 - ($suma % 11);
  
  if ($dvr == 11) $dvr = 0;
  if ($dvr == 10) $dvr = 'K';
  if ($dvr == $dv) return true;

  return false;
}

function calcularDígitoVerificador($nroRut) {
	$i = 2;
    $suma = 0;
	
    foreach (array_reverse(str_split($nroRut)) as $v) {
        if ($i == 8) {
            $i = 2;
		}

        $suma += $v * $i;
        $i++;
    }

    $dvr = 11 - ($suma % 11);
    
    if ($dvr == 11) {
        $dvr = 0;
	}
    if ($dvr == 10) {
        $dvr = 'K';
	}

    return $dvr;
}

function validarSintaxisPasaporte($rut) {
	$paréntesisInicial = strpos($rut, '(');
	$paréntesisFinal = strpos($rut, ')');
	
	if ($paréntesisInicial === false || $paréntesisFinal === false) {
		return false;
	}
	
	$pasaporte = substr($rut, 0, $paréntesisInicial);
	$país = substr($rut, $paréntesisInicial + 1, $paréntesisFinal - $paréntesisInicial - 1);
	
	if (trimPersonalizado($pasaporte) === '' || trimPersonalizado($país) == '') {
		return false;
	}
	
	return true;
}

function normalizarTextoSinTildes($texto) {
	$texto = normalizarTexto($texto);
	$texto = str_replace(['Á', 'É', 'Í', 'Ó', 'Ú'], ['A', 'E', 'I', 'O', 'U'], $texto);
	
	return $texto;
}

function validarPasaporte($pasaporte) {
	global $países, $aliasPaíses;
	
	$paréntesisInicial = strpos($pasaporte, '(');
	$paréntesisFinal = strpos($pasaporte, ')');
	
	$país = substr($pasaporte, $paréntesisInicial + 1, $paréntesisFinal - $paréntesisInicial - 1);
	
	$país = normalizarTextoSinTildes($país);
	
	if ($países[$país] === null) {
		$alias = $aliasPaíses[$país];
		if ($alias === null || $países[$alias] === null) {		
			return false;
		}
	}
	
	return true;
}

function validarCeldaRut($hoja, $i) {
	$coordenada = 'A' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$rut = $celda->getValue();
	
	$rutLimpio = limpiarRut($rut);
	
	if (validarSintaxisRut($rutLimpio)) {
		if (!validarRut($rutLimpio)) {
			return [$coordenada => 'RUT inválido'];
		}
		return true;
	} elseif (validarSintaxisPasaporte($rut)) {
		if (!validarPasaporte($rut)) {
			return [$coordenada => 'Pasaporte inválido'];
		}
		return true;
	}
	
	return [$coordenada => 'La celda no cumple con el formato de RUT o pasaporte'];
}

function normalizarTexto($texto) {
	$texto = trimPersonalizado($texto);
	$texto = mb_strtoupper($texto, 'UTF-8');
	
	return $texto;
}

function validarCeldaNombres($hoja, $i) {
	$coordenada = 'B' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$nombres = $celda->getValue();
	$nombres = normalizarTexto($nombres);
	
	if ($nombres === null || trimPersonalizado($nombres) === '') {
		return [$coordenada => 'Faltan los nombres'];
	}
	return true;
}

function validarCeldaApellidoPaterno($hoja, $i) {
	$coordenada = 'C' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$apellido = $celda->getValue();
	$apellido = normalizarTexto($apellido);
	
	if ($apellido === null || trimPersonalizado($apellido) === '') {
		return [$coordenada => 'Faltan el apellido paterno'];
	}
	return true;
}

function validarFecha($fecha) {
	$fecha = preg_replace('/[^\-0-9]/', '', $fecha);
	
	$resultado = preg_match('/^([0-9]{2})-([0-9]{2})-([0-9]{4})$/', $fecha);
	if ($resultado !== 1) {
		return false;		
	}
	
	$arreglo = explode('-', $fecha);
	if (sizeof($arreglo) != 3) {
		return false;
	}
	
	return checkdate($arreglo[1], $arreglo[0], $arreglo[2]);
}

function validarCeldaFecha($hoja, $i) {
	$coordenada = 'E' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$fecha = $celda->getValue();
	
	if ($fecha === null || trimPersonalizado($fecha) === '') {
		return [$coordenada => 'Falta la fecha de nacimiento'];
	}
	
	$esFecha = PHPExcel_Shared_Date::isDateTime($celda);
	if ($esFecha && is_numeric($fecha)) {
		return true;
	}
	
	if (validarFecha($fecha)) {
		return true;
	}
	
	return [$coordenada => 'Fecha de nacimiento inválida'];
}

function validarCeldaGénero($hoja, $i) {
	$coordenada = 'F' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$género = $celda->getValue();
	
	if ($género === null || trimPersonalizado($género) === '') {
		return [$coordenada => 'Falta el género'];
	}
	
	$género = trimPersonalizado(mb_strtoupper($género, 'UTF-8'));
	if (in_array($género, ['F', 'M', 'FEMENINO', 'MASCULINO'])) {
		return true;
	}
	
	return [$coordenada => 'Género inválido'];
}

function obtenerPrevisiones() {
	global $mysqli;
	
	$sql_previsiones = "SELECT ID_EMPRESA, NBR FROM empresas WHERE FLAG_PREVISION = 'S'";
	
	$resultado_previsiones = $mysqli->query($sql_previsiones);
	
	$previsiones = [];
	
	while ($previsión = $resultado_previsiones->fetch_assoc()) {
		$nbr = normalizarTextoSinTildes($previsión['NBR']);
		$previsiones[$nbr] = $previsión['ID_EMPRESA'];
	}
	
	return $previsiones;
}

function validarCeldaPrevisión($hoja, $i, $esDerivada = false) {
	global $previsiones, $aliasPrevisiones;
	
	$coordenada = 'G' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$previsión = $celda->getValue();
	if ($previsión === null || trimPersonalizado($previsión) === '') return [$coordenada => 'Falta la previsión'];
	
	$previsión = normalizarTextoSinTildes($previsión);

	if ($previsiones[$previsión] !== null) return true; # id de empresa
	if ($esDerivada && $previsión == 'ISAPRE') $previsión = 'PARTICULAR'; // En caso de que sea una plantilla derivada, la previsión ISAPRE se considera como particular.
	
	$alias = $aliasPrevisiones[$previsión];
	if ($alias === null || $previsiones[$alias] === null) return [$coordenada => 'Previsión inválida'];
	# return [$coordenada => "Faaaaaaealta la previsión $previsión - $alias - {$previsiones[$alias]}"];
	
	return true;
}

function validarCeldaDirección($hoja, $i) {
	$coordenada = 'H' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$dirección = $celda->getValue();
	$dirección = normalizarTexto($dirección);
	
	if ($dirección === null || trimPersonalizado($dirección) === '') {
		return [$coordenada => 'Falta la dirección'];
	}
	return true;
}

function obtenerComunas() {
	global $mysqli;
	
	$sql_comunas = "SELECT CODIGO, DESCRIPCION_CG FROM codigos_generales WHERE COD_TABLA = 'SIS_COMUN'";
	
	$resultado_comunas = $mysqli->query($sql_comunas);
	
	$comunas = [];
	
	while ($comuna = $resultado_comunas->fetch_assoc()) {
		$nombre = normalizarTextoSinTildes($comuna['DESCRIPCION_CG']);
		$comunas[$nombre] = $comuna['CODIGO'];
	}
	
	return $comunas;
}

function obtenerPaíses() {
	global $mysqli;
	
	$sql_países = "SELECT CODIGO, DESCRIPCION_CG FROM codigos_generales WHERE COD_TABLA = 'SIS_PAIS'";
	
	$resultado_países = $mysqli->query($sql_países);
	
	$países = [];
	
	while ($país = $resultado_países->fetch_assoc()) {
		$nombre = normalizarTextoSinTildes($país['DESCRIPCION_CG']);
		$países[$nombre] = $país['CODIGO'];
	}
	
	return $países;
}

function validarCeldaComuna($hoja, $i) {
	global $comunas, $aliasComunas;
	
	$coordenada = 'I' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$comuna = $celda->getValue();
	
	if ($comuna === null || trimPersonalizado($comuna) === '') {
		return [$coordenada => 'Falta la comuna'];
	}
	
	$comuna = normalizarTextoSinTildes($comuna);
	if ($comunas[$comuna] === null) {
		$alias = $aliasComunas[$comuna];
		if ($alias === null || $comunas[$alias] === null) {		
			return [$coordenada => 'Comuna inválida'];
		}
	}
	
	return true;
}

function limpiarTeléfono($teléfono) {
	$teléfono = trimPersonalizado($teléfono);
	$teléfono = str_replace(' ', '', $teléfono);
	$teléfono = str_replace('+', '', $teléfono);
	
	return $teléfono;
}

function validarTeléfono($teléfono) {
	$teléfono = preg_replace('/[^0-9]/', '', $teléfono);
	
	$resultado = preg_match('/^([0-9]+)$/', $teléfono);
	if ($resultado === 1) return true;	
	return false;
}

function validarCeldaTeléfono($hoja, $i) {
	$coordenada = 'J' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$teléfono = $celda->getValue();
	
	if ($teléfono === null || trimPersonalizado($teléfono) === '') return [$coordenada => 'Falta el teléfono'];

	$teléfono = limpiarTeléfono($teléfono);
	if (validarTeléfono($teléfono)) return true;
	
	return [$coordenada => 'Teléfono inválido'];
}

function validarCorreo($correo) {
	$resultado = preg_match('/^([a-zA-Z0-9_.+-])+\@(([a-zA-Z0-9-])+\.)+([a-zA-Z0-9]{2,4})+$/', $correo);
	if ($resultado === 1) return true;
	return false;
}

function validarCeldaCorreo($hoja, $i) {
	$coordenada = 'K' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$correo = $celda->getValue();
	
	if ($correo === null || trimPersonalizado($correo) === '') return true;
	if (validarCorreo($correo))	return true;
	
	return [$coordenada => 'Correo inválido'];
}

function limpiarIdentificadorMuestra($id) {
	$id = trimPersonalizado($id);
	
	return $id;
}

function validarIdentificadorMuestra($id) {
	$resultado = preg_match('/^[0-9]+$/', $id);
	if ($resultado === 1) return true;
	return false;
}

function validarCeldaIdentificadorMuestra($hoja, $i) {
	$coordenada = 'L' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$id = $celda->getValue();
	
	if ($id === null || trimPersonalizado($id) === '') return [$coordenada => 'Falta el identificador de muestra'];

	$id = limpiarIdentificadorMuestra($id);

	if (!validarIdentificadorMuestra($id)) return [$coordenada => 'Identificador de muestra inválido'];

	$datos_muestra = obtenerDatosMuestra($id);
	if (isset($datos_muestra['error'])) {
		return [$coordenada => $datos_muestra['error']];
	}
	
	if ($datos_muestra === false) {
		return [$coordenada => 'Ocurrió un error al obtener los datos de la muestra desde el MINSAL.'];
	}
	
	$identificación = obtenerIdentificación($hoja, $i);
	if (!$identificación['es_extranjero']) {
		if ($datos_muestra['paciente_tipodoc'] != 'RUN') {
			return [$coordenada => 'El paciente figura como extranjero en el MINSAL, pero en la plantilla figura como nacional.'];
		}
		if (strcasecmp($datos_muestra['id_paciente'], $identificación['nro_rut'] . '-' . $identificación['dv_rut']) != 0) {
			return [$coordenada => 'La identificación del paciente en el MINSAL no coincide con los datos de la plantilla. Datos MINSAL: ' . $datos_muestra['id_paciente'] . '.'];
		}
	} else {
		if ($datos_muestra['paciente_tipodoc'] == 'RUN') {
			return [$coordenada => 'El paciente figura como nacional en el MINSAL, pero en la plantilla figura como extranjero.'];
		}
		if (strcasecmp($datos_muestra['id_paciente'], $identificación['pasaporte']) != 0) {
			return [$coordenada => 'La identificación del paciente en el MINSAL no coincide con los datos de la plantilla. Datos MINSAL: ' . $datos_muestra['id_paciente'] . '.'];
		}
	}
	
	return true;
}

function validarCeldaFechaTomaMuestra($hoja, $i) {
	$coordenada = 'N' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$fechaTm = $celda->getValue();
	
	// $codigo_formato = $celda->getWorksheet()->getStyle($celda->getCoordinate())->getNumberFormat()->getFormatCode();
	// return [$coordenada => "$codigo_formato"];

	if ($fechaTm === null || trimPersonalizado($fechaTm) === '') return true;

	$esFecha = PHPExcel_Shared_Date::isDateTime($celda);
	if ($esFecha && is_numeric($fechaTm)) return true;

	$fechaTm = preg_replace('/[^\-0-9]/', '', $fechaTm);
	$resultado = preg_match('/^([0-9]{2})-([0-9]{2})-([0-9]{4})$/', $fechaTm);
	if ($resultado !== 1) return [$coordenada => 'Fecha de toma de muestra con formato inválido'];
	
	$arreglo = explode('-', $fechaTm);
	if (sizeof($arreglo) != 3) return [$coordenada => 'Fecha de toma de muestra con formato inválido'];
	if (!checkdate($arreglo[1], $arreglo[0], $arreglo[2])) return [$coordenada => 'Fecha de toma de muestra inválida (fecha no existe)'];
	
	return true;
}

function validarCeldaHoraTomaMuestra($hoja, $i) {
	$coordenada = 'M' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$horatTm = $celda->getValue();

	if ($horatTm === null || trimPersonalizado($horatTm) === '') return true;
	
	$codigo_formato = $celda->getWorksheet()->getStyle($celda->getCoordinate())->getNumberFormat()->getFormatCode();
	if ($codigo_formato != "h:mm") return [$coordenada => "Formato de celda incorrecto $codigo_formato"];
	# else return [$coordenada => " - $codigo_formato - $horatTm"];
	return true;
}

function validarHoja($hoja, $cantidadPacientes, $esDerivada = false) {
	$errores = [];
	for ($i = 0; $i < $cantidadPacientes; $i++) {
		$erroresPaciente = [];
		
		$errorRut = validarCeldaRut($hoja, $i);
		if ($errorRut !== true) $erroresPaciente = array_merge($erroresPaciente, $errorRut);
		
		$errorNombres = validarCeldaNombres($hoja, $i);
		if ($errorNombres !== true) $erroresPaciente = array_merge($erroresPaciente, $errorNombres);
		
		$errorApellidoPaterno = validarCeldaApellidoPaterno($hoja, $i);
		if ($errorApellidoPaterno !== true) $erroresPaciente = array_merge($erroresPaciente, $errorApellidoPaterno);
		
		$errorFecha = validarCeldaFecha($hoja, $i);
		if ($errorFecha !== true) $erroresPaciente = array_merge($erroresPaciente, $errorFecha);
		
		$errorGénero = validarCeldaGénero($hoja, $i);
		if ($errorGénero !== true) $erroresPaciente = array_merge($erroresPaciente, $errorGénero);
		
		$errorPrevisión = validarCeldaPrevisión($hoja, $i, $esDerivada);
		if ($errorPrevisión !== true) $erroresPaciente = array_merge($erroresPaciente, $errorPrevisión);

		$errorDirección = validarCeldaDirección($hoja, $i);
		if ($errorDirección !== true) $erroresPaciente = array_merge($erroresPaciente, $errorDirección);
		
		$errorComuna = validarCeldaComuna($hoja, $i);
		if ($errorComuna !== true) $erroresPaciente = array_merge($erroresPaciente, $errorComuna);
		
		$errorTeléfono = validarCeldaTeléfono($hoja, $i);
		if ($errorTeléfono !== true) $erroresPaciente = array_merge($erroresPaciente, $errorTeléfono);

		$errorCorreo = validarCeldaCorreo($hoja, $i);
		if ($errorCorreo !== true) $erroresPaciente = array_merge($erroresPaciente, $errorCorreo);

		$errorFechaTomaMuestra = validarCeldaFechaTomaMuestra($hoja, $i);
		if ($errorFechaTomaMuestra !== true) $erroresPaciente = array_merge($erroresPaciente, $errorFechaTomaMuestra);
		
		$errorHoraTomaMuestra = validarCeldaHoraTomaMuestra($hoja, $i);
		if ($errorHoraTomaMuestra !== true) $erroresPaciente = array_merge($erroresPaciente, $errorHoraTomaMuestra);
		
		if ($esDerivada) {
			$errorIdentificadorMuestra = validarCeldaIdentificadorMuestra($hoja, $i);
			if ($errorIdentificadorMuestra !== true) $erroresPaciente = array_merge($erroresPaciente, $errorIdentificadorMuestra);
		}
		
		if (sizeof($erroresPaciente) > 0) $errores = array_merge($errores, $erroresPaciente);
	}
	
	return $errores;
}

function obtenerPaís($pasaporte) {
	global $países, $aliasPaíses;
	
	$paréntesisInicial = strpos($pasaporte, '(');
	$paréntesisFinal = strpos($pasaporte, ')');
	
	$país = substr($pasaporte, $paréntesisInicial + 1, $paréntesisFinal - $paréntesisInicial - 1);

	$país = normalizarTextoSinTildes($país);
	$id = $países[$país];
	if ($id !== null) {
		return $id;
	}
	
	return $países[$aliasPaíses[$país]];
}

function obtenerIdentificación($hoja, $i) {
	$coordenada = 'A' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$rut = $celda->getValue();
	$rutLimpio = limpiarRut($rut);

	if (validarSintaxisRut($rutLimpio)) {
		$arreglo = explode('-', $rutLimpio);
		$nro_rut = $arreglo[0];
		if (substr($nro_rut, 0, 1) === '0') $nro_rut = substr($nro_rut, 1);
		return ['es_extranjero' => false, 'nro_rut' => $nro_rut, 'dv_rut' => $arreglo[1]];
	} else {
		$pasaporte = substr($rut, 0, strpos($rut, '('));
		$pasaporte = normalizarTexto($pasaporte);
		$país = obtenerPaís($rut);
		return ['es_extranjero' => true, 'pasaporte' => $pasaporte, 'país' => $país];
	}
}

function obtenerNombres($hoja, $i) {
	$coordenada = 'B' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$nombres = $celda->getValue();
	$nombres = normalizarTexto($nombres);
	return $nombres;
}

function obtenerApellidoPaterno($hoja, $i) {
	$coordenada = 'C' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$apellido = $celda->getValue();
	$apellido = normalizarTexto($apellido);
	
	return $apellido;
}

function obtenerApellidoMaterno($hoja, $i) {
	$coordenada = 'D' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$apellido = $celda->getValue();
	
	if ($apellido === null || trimPersonalizado($apellido) === '') {
		return null;
	}
	
	$apellido = normalizarTexto($apellido);
	
	return $apellido;
}

function obtenerFecha($fecha) {
	date_default_timezone_set('GMT');
	$retorno = date('Y-m-d', PHPExcel_Shared_Date::ExcelToPHP($fecha));
	date_default_timezone_set('America/Santiago');
	return $retorno;
}

function obtenerFechaNacimiento($hoja, $i) {
	$coordenada = 'E' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$fecha = $celda->getValue();
	
	$esFecha = PHPExcel_Shared_Date::isDateTime($celda);
	if ($esFecha && is_numeric($fecha)) {
		return obtenerFecha($fecha);
	}
	
	$fecha = preg_replace('/[^\-0-9]/', '', $fecha);
	
	return implode('-', array_reverse(explode('-', $fecha)));
}

function obtenerGénero($hoja, $i) {
	$coordenada = 'F' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$género = $celda->getValue();
	
	$género = trimPersonalizado(mb_strtoupper($género, 'UTF-8'));
	
	return $género[0];
}

function obtenerPrevisión($hoja, $i, $esDerivada = false) {
	global $previsiones, $aliasPrevisiones;
	
	$coordenada = 'G' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$previsión = $celda->getValue();
	$previsión = normalizarTextoSinTildes($previsión);
	
	if ($previsiones[$previsión] !== null) return $previsiones[$previsión]; # id de empresa
	
	if ($esDerivada && $previsión == 'ISAPRE') $previsión = 'PARTICULAR'; 	// En caso de que sea una plantilla derivada, la previsión ISAPRE se considera como particular.
	$alias = $aliasPrevisiones[$previsión];
	
	return $previsiones[$alias];
}

function obtenerDirección($hoja, $i) {
	global $comunas, $aliasComunas;
	
	$coordenada = 'H' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$dirección = $celda->getValue();
	$dirección = normalizarTexto($dirección);
	
	return $dirección;
}

function obtenerComuna($hoja, $i) {
	global $comunas, $aliasComunas;
	
	$coordenada = 'I' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$comuna = $celda->getValue();
	$comuna = normalizarTextoSinTildes($comuna);
	
	$id = $comunas[$comuna];
	if ($id !== null) {
		return $id;
	}
	
	return $comunas[$aliasComunas[$comuna]];
}

function obtenerTeléfono($hoja, $i) {
	
	$coordenada = 'J' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$teléfono = $celda->getValue();
	$teléfono = limpiarTeléfono($teléfono);
	return $teléfono;
}

function obtenerCorreo($hoja, $i) {
	
	$coordenada = 'K' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$correo = $celda->getValue();
	if (trimPersonalizado($correo) === '') return null;
	return $correo;
}

function obtenerIdentificadorMuestra($hoja, $i) {
	
	$coordenada = 'L' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$id = $celda->getValue();
	$id = trimPersonalizado($id);
	return $id;
}

function obtenerFechaTomaMuestra($hoja, $i) {
	$coordenada = 'N' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$fechaTm = $celda->getValue();
	
	if ($fechaTm === null || trimPersonalizado($fechaTm) === '') return null;

	$esFecha = PHPExcel_Shared_Date::isDateTime($celda);
	if ($esFecha && is_numeric($fechaTm)) return obtenerFecha($fechaTm);
	
	$fechaTm = preg_replace('/[^\-0-9]/', '', $fechaTm);
	return implode('-', array_reverse(explode('-', $fechaTm)));
}

function obtenerHoraTomaMuestra($hoja, $i) {
	$coordenada = 'M' . ($i + 2);
	$celda = $hoja->getCell($coordenada);
	$horaTm = $celda->getCalculatedValue();
	
	if ($horaTm === null || trimPersonalizado($horaTm) === '') return null;
	
	$horaTm = PHPExcel_Style_NumberFormat::toFormattedString($horaTm, 'hh:mm:ss');
	return $horaTm;
}

function obtenerPacientesExcel($hoja, $cantidadPacientes, $esDerivada = false) {
	$pacientes = [];
	
	for ($i = 0; $i < $cantidadPacientes; $i++) {
		$paciente = [];
		
		$paciente['identificación'] = obtenerIdentificación($hoja, $i);
		$paciente['nombres'] = obtenerNombres($hoja, $i);
		$paciente['apellido_paterno'] = obtenerApellidoPaterno($hoja, $i);
		$paciente['apellido_materno'] = obtenerApellidoMaterno($hoja, $i);
		$paciente['fecha_nacimiento'] = obtenerFechaNacimiento($hoja, $i);
		$paciente['género'] = obtenerGénero($hoja, $i);
		$paciente['previsión'] = obtenerPrevisión($hoja, $i, $esDerivada);
		$paciente['dirección'] = obtenerDirección($hoja, $i);
		$paciente['comuna'] = obtenerComuna($hoja, $i);
		$paciente['teléfono'] = obtenerTeléfono($hoja, $i);
		$paciente['correo'] = obtenerCorreo($hoja, $i);
		$paciente['fecha_tm'] = obtenerFechaTomaMuestra($hoja, $i);
		$paciente['hora_tm'] = obtenerHoraTomaMuestra($hoja, $i);
		
		if ($esDerivada) {
			$paciente['id_muestra'] = obtenerIdentificadorMuestra($hoja, $i);
		}
		
		if (!$paciente['identificación']['es_extranjero']) {
			if ($pacientes[$paciente['identificación']['nro_rut']] !== null) {
				$rutCompleto = $paciente['identificación']['nro_rut'] . '-' . $paciente['identificación']['dv_rut'];
				throw new Exception("El RUT $rutCompleto se encuentra repetido en el archivo.");
			}
			$pacientes[$paciente['identificación']['nro_rut']] = $paciente;
		} else {
			$pacientes[$paciente['identificación']['pasaporte']] = $paciente;
		}
	}
	
	return $pacientes;
}

function obtenerPacientesBaseDatos($pacientesExcel) {
	global $mysqli;
	
	$ruts = [];
	foreach ($pacientesExcel as $paciente) {
		if (!$paciente['identificación']['es_extranjero']) {
			$ruts[] = $paciente['identificación']['nro_rut'];
		}
	}
	
	if (sizeof($ruts) === 0) {
		return [];
	}
	
	$sql_pacientes = 'SELECT ID_PERSONA AS id_persona, NRO_RUT AS nro_rut, DV_RUT AS dv_rut, NOMBRES AS nombres, APELLIDO_PATERNO AS apellido_paterno FROM personas WHERE NRO_RUT IN (' . implode(',', $ruts) . ')';
	
	$resultado_pacientes = $mysqli->query($sql_pacientes);
	
	$pacientesBaseDatos = [];
	while($paciente = $resultado_pacientes->fetch_assoc()) {
		$paciente['nombres'] = normalizarTexto($paciente['nombres']);
		$paciente['apellido_paterno'] = normalizarTexto($paciente['apellido_paterno']);
		
		$pacientesBaseDatos[$paciente['nro_rut']] = $paciente;
	}	
	
	return $pacientesBaseDatos;
}

function obtenerDiferencias($pacientesBaseDatos, $pacientesExcel) {
	$errores = [];
	
	$i = 1;
	foreach ($pacientesExcel as $rut => $pacienteExcel) {
		$i++;
		
		if ($pacienteExcel['identificación']['es_extranjero']) {
			continue;
		}
		
		$pacienteBaseDatos = $pacientesBaseDatos[$rut];
		if ($pacienteBaseDatos === null) {
			continue;
		}

		$nombreExcel = explode(' ', $pacienteExcel['nombres'])[0];
		$nombreBaseDatos = explode(' ', $pacienteBaseDatos['nombres'])[0];
		
		$apellidoExcel = $pacienteExcel['apellido_paterno'];
		$apellidoBaseDatos = $pacienteBaseDatos['apellido_paterno'];
		
		$nombreCompletoExcel = $nombreExcel . ' ' . $apellidoExcel;
		$nombreCompletoBaseDatos = $nombreBaseDatos . ' ' . $apellidoBaseDatos;
		
		$porcentaje;
		similar_text($nombreCompletoExcel, $nombreCompletoBaseDatos, $porcentaje);
		
		if ($porcentaje < 90) {
			$coordenada = 'A' . $i;
			$rut = $rut . '-' . $pacienteExcel['identificación']['dv_rut'];
			$mensaje = "El paciente con RUT $rut ya se encuentra registrado como $nombreCompletoBaseDatos. Verifique los datos del paciente original y del nuevo paciente ($nombreCompletoExcel)";
			
			$error = [$coordenada => $mensaje];
			$errores = array_merge($errores, $error);
		}
	}
	
	return $errores;
}

function grabarPacientes($pacientesBaseDatos, $pacientesExcel, $id_convenio, $esDerivada = false) {
	global $mysqli;
	
	$datos_atencion = [];
	$i = 0;

	foreach ($pacientesExcel as $pacienteExcel) {
		$correoPaciente = null;
		if ($id_convenio == "0") $correoPaciente = $pacienteExcel['correo'];

		if (!$pacienteExcel['identificación']['es_extranjero']) {
			$pacienteBaseDatos = $pacientesBaseDatos[$pacienteExcel['identificación']['nro_rut']];   
			if ($pacienteBaseDatos === null) {
				$sentencia = $mysqli->prepare("INSERT INTO personas (NRO_RUT, DV_RUT, NOMBRES, APELLIDO_PATERNO, APELLIDO_MATERNO, FECHA_NACIMIENTO, GENERO, ID_PREVISION, DIRECCION, COMUNA, FONO_1, MAIL, RAZA, FLAG_PACIENTE, FLAG_MEDICO, ESPECIALIDAD) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, '1', 'S', 'N', '-')");
				$sentencia->bind_param('issssssisiss', $pacienteExcel['identificación']['nro_rut'], $pacienteExcel['identificación']['dv_rut'], $pacienteExcel['nombres'], $pacienteExcel['apellido_paterno'], $pacienteExcel['apellido_materno'], $pacienteExcel['fecha_nacimiento'], $pacienteExcel['género'], $pacienteExcel['previsión'], $pacienteExcel['dirección'], $pacienteExcel['comuna'], $pacienteExcel['teléfono'], $correoPaciente);
				$sentencia->execute();	
				$datos_atencion[$i]["id_persona"] = $sentencia->insert_id;
				$datos_atencion[$i]["id_muestra"] = $esDerivada ? $pacienteExcel['id_muestra'] : null;
				$datos_atencion[$i]["fecha_tm"] = $pacienteExcel["fecha_tm"];
				$datos_atencion[$i]["hora_tm"] = $pacienteExcel["hora_tm"];
				$sentencia->close();
			} else {
				$dv = calcularDígitoVerificador($pacienteExcel['identificación']['nro_rut']);
				$sentencia = $mysqli->prepare("UPDATE personas SET DV_RUT = ?, FECHA_NACIMIENTO = ?, ID_PREVISION = ?, DIRECCION = ?, COMUNA = ?, FONO_1 = ?, MAIL = ?, FLAG_PACIENTE = 'S' WHERE ID_PERSONA = ? LIMIT 1");
				$sentencia->bind_param('ssisissi', $dv, $pacienteExcel['fecha_nacimiento'], $pacienteExcel['previsión'], $pacienteExcel['dirección'], $pacienteExcel['comuna'],
        $pacienteExcel['teléfono'], $correoPaciente, $pacienteBaseDatos['id_persona']);
				$sentencia->execute();
				$sentencia->close();
				$datos_atencion[$i]["id_persona"] = $pacienteBaseDatos['id_persona'];
				$datos_atencion[$i]["id_muestra"] = $esDerivada ? $pacienteExcel['id_muestra'] : null;
				$datos_atencion[$i]["fecha_tm"] = $pacienteExcel["fecha_tm"];
				$datos_atencion[$i]["hora_tm"] = $pacienteExcel["hora_tm"];
			}
		} else {
			$sentenciaCreación = $mysqli->prepare("INSERT INTO personas (NRO_RUT, DV_RUT, PASAPORTE, PAIS, NOMBRES, APELLIDO_PATERNO, APELLIDO_MATERNO, FECHA_NACIMIENTO, GENERO, ID_PREVISION, DIRECCION, COMUNA, FONO_1, MAIL, RAZA, FLAG_PACIENTE, FLAG_MEDICO, ESPECIALIDAD) VALUES (obtener_rut_extranjero(), 0, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, '1', 'S', 'N', '-')");
			$sentenciaCreación->bind_param('sisssssisiss', $pacienteExcel['identificación']['pasaporte'], $pacienteExcel['identificación']['país'], $pacienteExcel['nombres'], $pacienteExcel['apellido_paterno'], $pacienteExcel['apellido_materno'], $pacienteExcel['fecha_nacimiento'], $pacienteExcel['género'], $pacienteExcel['previsión'], $pacienteExcel['dirección'], $pacienteExcel['comuna'], $pacienteExcel['teléfono'], $correoPaciente);
			$sentenciaCreación->execute();
			$id = $sentenciaCreación->insert_id;
			$sentenciaCreación->close();
			$consulta = $mysqli->query("SELECT NRO_RUT FROM personas WHERE ID_PERSONA = $id LIMIT 1");
			$nroRut = $consulta->fetch_assoc()['NRO_RUT'];
			$dvRut = calcularDígitoVerificador($nroRut);
			$sentenciaActualización = $mysqli->prepare("UPDATE personas SET DV_RUT = ? WHERE ID_PERSONA = ? LIMIT 1");
			$sentenciaActualización->bind_param('si', $dvRut, $id);
			$sentenciaActualización->execute();
			$sentenciaActualización->close();
			$datos_atencion[$i]["id_persona"] = $id;
			$datos_atencion[$i]["id_muestra"] = $esDerivada ? $pacienteExcel['id_muestra'] : null;
			$datos_atencion[$i]["fecha_tm"] = $pacienteExcel["fecha_tm"];
			$datos_atencion[$i]["hora_tm"] = $pacienteExcel["hora_tm"];
		}
		$i++;
	}
	
	return $datos_atencion;
}

function crearPlantilla($datos_atencion, $nombre, $id_convenio, $id_procedencia, $obs, $esDerivada = false) {
	global $mysqli;
	
	$id_medico = 64475;
	$cod_urgencia = 1;
	$es_derivada = $esDerivada ? 1 : null;
	
	$sentencia = $mysqli->prepare("INSERT INTO plantillas (nombre, id_convenio, id_local, id_medico, cod_urgencia, obs, es_derivada) VALUES (?, ?, ?, ?, ?, ?, ?)");
	$sentencia->bind_param('siiiisi', $nombre, $id_convenio, $id_procedencia, $id_medico, $cod_urgencia, $obs, $es_derivada);
	$sentencia->execute();	
	$id_plantilla = $sentencia->insert_id;
	$sentencia->close();
	
	foreach ($datos_atencion as $d) {
		// print_r($d);
		// echo "\n";
		$sentencia = $mysqli->prepare("INSERT INTO plantilla_pacientes (id_plantilla, id_paciente, id_muestra, fecha_tm, hora_tm) VALUES (?, ?, ?, ?, ?)");
		$sentencia->bind_param('iiiss', $id_plantilla, $d["id_persona"], $d["id_muestra"], $d["fecha_tm"], $d["hora_tm"]);
		$sentencia->execute();
		$sentencia->close();
	}
	
	$cod_pcr = '00.00.069';
	$sentencia = $mysqli->prepare("INSERT INTO plantilla_examenes (id_plantilla, cod_interno) VALUES (?, ?)");
	$sentencia->bind_param('is', $id_plantilla, $cod_pcr);
	$sentencia->execute();			
	$sentencia->close();
	
	return $id_plantilla;
}

function obtenerDatosMuestra($id) {
	global $url_api, $clave_de_acceso_api;
	
	$json_solicitud = [
		'id_muestra' => $id
	];
	
	$cliente = new Client(
		[
			'headers' => [
				'Content-Type' => 'multipart/form-data',
				'ACCESSKEY' => $clave_de_acceso_api
			]
		]
	);
	
	try {	
		$respuesta = $cliente->post(
			$url_api . '/datosMuestraID',
			[
				'multipart' => [
					[
						'name'     => 'parametros',
						'contents' => json_encode($json_solicitud)
					]
				]
			]
		);

		$objeto_respuesta = json_decode($respuesta->getBody()->getContents(), true);
		if ($objeto_respuesta === null) {
			return ['error' => 'Ocurrió un error al obtener los datos de la muestra ' . $id . ': ' . $respuesta->getBody()->getContents()];
		}
		
		if (sizeof($objeto_respuesta) === 0) {
			return ['error' => 'Ocurrió un error al obtener los datos de la muestra ' . $id . '. Verifique que la muestra haya sido enviada al laboratorio.'];
		}
		
		if (!isset($objeto_respuesta[0]['id_paciente'])) {
			return ['error' => 'Ocurrió un error al obtener los datos de la muestra ' . $id . '. La respuesta del MINSAL es inválida.'];
		}
		
		return $objeto_respuesta[0];
	} catch (ServerException $e) {		
		return ['error' => 'Ocurrió un error al obtener los datos de la muestra ' . $id . ': ' . $e->getResponse()->getBody()->getContents()];
	} catch (ClientException $e) {
		return ['error' => 'Ocurrió un error al obtener los datos de la muestra ' . $id . ': ' . $e->getResponse()->getBody()->getContents()];
	} catch (ConnectException $e) {		
		return ['error' => 'Ocurrió un error al obtener los datos de la muestra ' . $id . '. No se pudo conectar con el MINSAL.'];
	}
	
	return false;
}

function obtenerDatosEmpresa($id) {
	global $mysqli;
	
	$sql = "SELECT * FROM empresas WHERE ID_EMPRESA = $id";
	$resultado = $mysqli->query($sql);
	$datos = $resultado->fetch_assoc();
	
	return $datos;
}

function obtenerDatosProcedencia($id) {
	global $mysqli;
	
	$sql = "SELECT * FROM centros_contacto WHERE ID_CENTRO_CONTACTO = $id";
	$resultado = $mysqli->query($sql);
	$datos = $resultado->fetch_assoc();
	
	return $datos;
}

?>